<?php
/**
 * API REST simple pour le calendrier des événements
 * 
 * Endpoints:
 * GET /api.php?action=events - Récupère tous les événements
 * GET /api.php?action=events&scope=national - Filtre par portée
 * POST /api.php?action=create - Crée un événement (à implémenter)
 */

header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, POST, PUT, DELETE, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

// Gestion des requêtes OPTIONS (CORS)
if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    http_response_code(200);
    exit;
}

// Chemin vers le fichier de données
$dataFile = __DIR__ . '/events.json';

/**
 * Charge les événements depuis le fichier JSON
 */
function loadEvents() {
    global $dataFile;
    if (!file_exists($dataFile)) {
        return [];
    }
    $json = file_get_contents($dataFile);
    return json_decode($json, true) ?: [];
}

/**
 * Sauvegarde les événements dans le fichier JSON
 */
function saveEvents($events) {
    global $dataFile;
    $json = json_encode($events, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    file_put_contents($dataFile, $json);
}

/**
 * Filtre les événements selon les critères
 */
function filterEvents($events, $filters) {
    if (empty($filters)) {
        return $events;
    }
    
    return array_filter($events, function($event) use ($filters) {
        // Filtre par portée
        if (isset($filters['scope']) && $filters['scope'] !== 'all') {
            if ($event['scope'] !== $filters['scope']) {
                return false;
            }
        }
        
        // Filtre par catégorie
        if (isset($filters['category'])) {
            if ($event['category'] !== $filters['category']) {
                return false;
            }
        }
        
        // Filtre par organisation
        if (isset($filters['organization'])) {
            if (!isset($event['organization']) || $event['organization'] !== $filters['organization']) {
                return false;
            }
        }
        
        return true;
    });
}

/**
 * Répond avec du JSON
 */
function respond($data, $status = 200) {
    http_response_code($status);
    echo json_encode($data, JSON_PRETTY_PRINT | JSON_UNESCAPED_UNICODE);
    exit;
}

/**
 * Répond avec une erreur
 */
function respondError($message, $status = 400) {
    respond(['error' => $message], $status);
}

// Router simple
$action = $_GET['action'] ?? 'events';
$method = $_SERVER['REQUEST_METHOD'];

switch ($action) {
    case 'events':
        if ($method !== 'GET') {
            respondError('Méthode non autorisée', 405);
        }
        
        $events = loadEvents();
        
        // Application des filtres
        $filters = [];
        if (isset($_GET['scope'])) {
            $filters['scope'] = $_GET['scope'];
        }
        if (isset($_GET['category'])) {
            $filters['category'] = $_GET['category'];
        }
        if (isset($_GET['organization'])) {
            $filters['organization'] = $_GET['organization'];
        }
        
        $filteredEvents = filterEvents($events, $filters);
        
        respond([
            'success' => true,
            'count' => count($filteredEvents),
            'events' => array_values($filteredEvents)
        ]);
        break;
    
    case 'create':
        if ($method !== 'POST') {
            respondError('Méthode non autorisée', 405);
        }
        
        $input = json_decode(file_get_contents('php://input'), true);
        
        // Validation basique
        if (empty($input['title']) || empty($input['start'])) {
            respondError('Titre et date de début requis');
        }
        
        $events = loadEvents();
        
        // Génération d'un ID
        $maxId = 0;
        foreach ($events as $event) {
            if ($event['id'] > $maxId) {
                $maxId = $event['id'];
            }
        }
        
        $newEvent = [
            'id' => $maxId + 1,
            'title' => $input['title'],
            'start' => $input['start'],
            'end' => $input['end'] ?? null,
            'backgroundColor' => $input['backgroundColor'] ?? '#3b82f6',
            'borderColor' => $input['borderColor'] ?? '#2563eb',
            'scope' => $input['scope'] ?? 'local',
            'category' => $input['category'] ?? 'other',
            'organization' => $input['organization'] ?? null,
            'description' => $input['description'] ?? null
        ];
        
        $events[] = $newEvent;
        saveEvents($events);
        
        respond([
            'success' => true,
            'message' => 'Événement créé avec succès',
            'event' => $newEvent
        ], 201);
        break;
    
    case 'update':
        if ($method !== 'PUT') {
            respondError('Méthode non autorisée', 405);
        }
        
        $input = json_decode(file_get_contents('php://input'), true);
        $id = intval($_GET['id'] ?? 0);
        
        if ($id <= 0) {
            respondError('ID invalide');
        }
        
        $events = loadEvents();
        $found = false;
        
        foreach ($events as &$event) {
            if ($event['id'] === $id) {
                // Mise à jour des champs
                foreach ($input as $key => $value) {
                    if ($key !== 'id') {
                        $event[$key] = $value;
                    }
                }
                $found = true;
                break;
            }
        }
        
        if (!$found) {
            respondError('Événement non trouvé', 404);
        }
        
        saveEvents($events);
        
        respond([
            'success' => true,
            'message' => 'Événement mis à jour avec succès'
        ]);
        break;
    
    case 'delete':
        if ($method !== 'DELETE') {
            respondError('Méthode non autorisée', 405);
        }
        
        $id = intval($_GET['id'] ?? 0);
        
        if ($id <= 0) {
            respondError('ID invalide');
        }
        
        $events = loadEvents();
        $initialCount = count($events);
        
        $events = array_filter($events, function($event) use ($id) {
            return $event['id'] !== $id;
        });
        
        if (count($events) === $initialCount) {
            respondError('Événement non trouvé', 404);
        }
        
        saveEvents(array_values($events));
        
        respond([
            'success' => true,
            'message' => 'Événement supprimé avec succès'
        ]);
        break;
    
    case 'stats':
        if ($method !== 'GET') {
            respondError('Méthode non autorisée', 405);
        }
        
        $events = loadEvents();
        
        $stats = [
            'total' => count($events),
            'by_scope' => [],
            'by_category' => []
        ];
        
        foreach ($events as $event) {
            // Stats par portée
            $scope = $event['scope'];
            if (!isset($stats['by_scope'][$scope])) {
                $stats['by_scope'][$scope] = 0;
            }
            $stats['by_scope'][$scope]++;
            
            // Stats par catégorie
            $category = $event['category'];
            if (!isset($stats['by_category'][$category])) {
                $stats['by_category'][$category] = 0;
            }
            $stats['by_category'][$category]++;
        }
        
        respond([
            'success' => true,
            'stats' => $stats
        ]);
        break;
    
    default:
        respondError('Action inconnue', 404);
}
